
function basemap (latmin,latrange,lonmin,lonrange,type_map,color)
%
% FUNCTION [] = basemap (latmin,latrange,lonmin,lonrange,type_map,color)
%
% Create a basemap of the world coastlines. Note that latitude
% and longitude RANGES are specified, rather than maximum values.
% If type_map=0, only coastlines are drawn. For type_map equal
% to 1, a slightly lower resolution shaded map is drawn. The
% coastlines and fill color is blue by default, but can be changed
% with the input string variable, color.
%
 
 
% INPUTS :    latmin - The southernmost latitude. South latitude
%                      is taken to be negative.
%                      
%           latrange - The range of latitude to cover. The
%                      maximum latitude is computed as
%                      LATMIN + LATRANGE.
%                      
%             lonmin - The westernmost longitude. West longitude
%                      is taken to be negative.
%                      
%           lonrange - Analogous to LATRANGE.
%                      
%           type_map - Controls the type of map drawn. The legal
%                      values are 0 or 1, which result in:
%                      0 - Only the coastlines
%                      1 - A lower resolution shaded map
%
%              color - String variable giving color to use for coastlines
%                      and land fill. Default is blue.
 
 
 
% HISTORY:  1. Version 1 written 27 June 1990 by Gary Mitchum
%              
 
% 1234567890123456789012345678901234567890123456789012345678901234567890


% Do some argument checking.

if nargin<5; type_map=0; color='b'; end
if nargin<6; color='b'; end
if type_map<0 | type_map>1; type_map=0; end

load basemap

latmax=latmin+latrange;
if latmin<-90; latmin=-90; end; if latmax>90; latmax=90; end
latmin=5*round(latmin/5); latmax=5*round(latmax/5);
latrange=latmax-latmin; if latrange<10; latmax=latmin+10; latrange=10; end

if lonmin<-360; lonmin=-360; end; lonmax=lonmin+lonrange;
if lonmax<0; lonmin = lonmin+360; lonmax = lonmax+360; end
lonmin=5*round(lonmin/5); lonmax=5*round(lonmax/5);
lonrange=lonmax-lonmin; if lonrange<10; lonmax=lonmin+10; lonrange=10; end


% Do the coastlines map.


if type_map==0

   [m,n]=size(lon);
   lon=[lon;NaN*ones(1,n)]; lon=lon(:);
   lat=[lat;NaN*ones(1,n)]; lat=lat(:);

   plot(lon,lat,'-','color',color); hold on
   if lonmin<-180; plot(lon-360,lat,'-','color',color); end
   if lonmax>180; plot(lon+360,lat,'-','color',color); end
   hold off

end

% Do the solid map.

if type_map==1

   for n=1:93
      eval(['x=x',int2str(n),'; y=y',int2str(n),';'])
      x=[x(:);x(1)]; y=[y(:);y(1)];
      fill(x,y,color);
      if n==1; hold on; end
      plot(x,y,'-','color',color)
      if lonmin<-180; fill(x-360,y,color); plot(x-360,y,'-','color',color); end
      if lonmax>180; fill(x+360,y,color); plot(x+360,y,'-','color',color); end
   end

   hold off

end


% Set up the axis system.

axis([lonmin lonmax latmin latmax])
